/*
 * Decompiled with CFR 0.152.
 */
package org.tinymediamanager.core.movie;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.ser.DefaultSerializerProvider;
import com.fasterxml.jackson.module.blackbird.BlackbirdModule;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.lang3.StringUtils;
import org.h2.mvstore.MVMap;
import org.h2.mvstore.MVStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tinymediamanager.Globals;
import org.tinymediamanager.core.CustomNullStringSerializerProvider;
import org.tinymediamanager.core.ITmmModule;
import org.tinymediamanager.core.NullKeySerializer;
import org.tinymediamanager.core.Settings;
import org.tinymediamanager.core.TmmResourceBundle;
import org.tinymediamanager.core.Utils;
import org.tinymediamanager.core.entities.MediaEntity;
import org.tinymediamanager.core.http.TmmHttpServer;
import org.tinymediamanager.core.movie.MovieList;
import org.tinymediamanager.core.movie.MovieSettings;
import org.tinymediamanager.core.movie.entities.Movie;
import org.tinymediamanager.core.movie.entities.MovieSet;
import org.tinymediamanager.core.movie.http.MovieCommandHandler;

public final class MovieModuleManager
implements ITmmModule {
    private static final String MODULE_TITLE = "Movie management";
    private static final String MOVIE_DB = "movies.db";
    private static final Logger LOGGER = LoggerFactory.getLogger(MovieModuleManager.class);
    private static final int COMMIT_DELAY = 2000;
    private static MovieModuleManager instance;
    private final List<String> startupMessages = new ArrayList<String>();
    private final Map<MediaEntity, Long> pendingChanges = new HashMap<MediaEntity, Long>();
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private boolean enabled = false;
    private int autoCommitBufferSize = 8192;
    private MVStore mvStore;
    private ObjectWriter movieObjectWriter;
    private ObjectReader movieObjectReader;
    private ObjectWriter movieSetObjectWriter;
    private ObjectReader movieSetObjectReader;
    private MVMap<UUID, String> movieMap;
    private MVMap<UUID, String> movieSetMap;
    private Timer databaseTimer;

    private MovieModuleManager() {
        int bufferSize = Integer.getInteger("tmm.mvstore.buffersize", 8);
        if (2 <= bufferSize && bufferSize <= 64) {
            this.autoCommitBufferSize = 1024 * bufferSize;
        }
    }

    public static MovieModuleManager getInstance() {
        if (instance == null) {
            instance = new MovieModuleManager();
        }
        return instance;
    }

    static void clearInstances() {
        instance = null;
        MovieSettings.clearInstance();
        MovieList.clearInstance();
    }

    public MovieSettings getSettings() {
        return MovieSettings.getInstance();
    }

    public MovieList getMovieList() {
        return MovieList.getInstance();
    }

    @Override
    public String getModuleTitle() {
        return MODULE_TITLE;
    }

    @Override
    public void startUp() {
        ObjectMapper objectMapper = ((JsonMapper.Builder)((JsonMapper.Builder)((JsonMapper.Builder)((JsonMapper.Builder)((JsonMapper.Builder)((JsonMapper.Builder)JsonMapper.builder().configure(MapperFeature.AUTO_DETECT_GETTERS, false)).configure(MapperFeature.AUTO_DETECT_IS_GETTERS, false)).configure(MapperFeature.AUTO_DETECT_SETTERS, false)).configure(MapperFeature.AUTO_DETECT_FIELDS, false)).configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)).addModule((Module)new BlackbirdModule())).build();
        objectMapper.setTimeZone(TimeZone.getDefault());
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_DEFAULT);
        objectMapper.setSerializerProvider((DefaultSerializerProvider)new CustomNullStringSerializerProvider());
        objectMapper.getSerializerProvider().setNullKeySerializer((JsonSerializer)new NullKeySerializer());
        this.movieObjectWriter = objectMapper.writerFor(Movie.class);
        this.movieObjectReader = objectMapper.readerFor(Movie.class);
        this.movieSetObjectWriter = objectMapper.writerFor(MovieSet.class);
        this.movieSetObjectReader = objectMapper.readerFor(MovieSet.class);
        this.openDatabaseAndLoadMovies();
        this.enabled = true;
        TimerTask databaseWriteTask = new TimerTask(){

            @Override
            public void run() {
                MovieModuleManager.this.writePendingChanges();
            }
        };
        this.databaseTimer = new Timer();
        this.databaseTimer.schedule(databaseWriteTask, 2000L, 2000L);
        try {
            TmmHttpServer.getInstance().createContext("movie", new MovieCommandHandler());
        }
        catch (Exception e) {
            LOGGER.warn("could not register movie API - '{}'", (Object)e.getMessage());
        }
    }

    private void openDatabaseAndLoadMovies() {
        Path databaseFile = Paths.get(Settings.getInstance().getSettingsFolder(), MOVIE_DB);
        try {
            this.loadDatabase(databaseFile);
            return;
        }
        catch (Exception e) {
            if (e instanceof IllegalStateException && e.getMessage().contains("file is locked")) {
                throw e;
            }
            if (this.mvStore != null && !this.mvStore.isClosed()) {
                this.mvStore.close();
            }
            LOGGER.error("Could not open database file: {}", (Object)e.getMessage());
            try {
                Utils.deleteFileSafely(Paths.get(Globals.BACKUP_FOLDER, "movies.db.corrupted"));
                Utils.moveFileSafe(databaseFile, Paths.get(Globals.BACKUP_FOLDER, "movies.db.corrupted"));
            }
            catch (Exception e2) {
                LOGGER.error("Could not move corrupted database to '{}' - '{}", (Object)"movies.db.corrupted", (Object)e2.getMessage());
            }
            LOGGER.info("try to restore the database from the backups");
            List<Path> backups = Utils.listFiles(Paths.get(Globals.BACKUP_FOLDER, new String[0]));
            backups.sort(Comparator.reverseOrder());
            boolean first = true;
            for (Path backup : backups) {
                if (!backup.getFileName().toString().startsWith("data.")) continue;
                if (first) {
                    first = false;
                    continue;
                }
                try {
                    Utils.unzipFile(backup, Paths.get("/", "data", MOVIE_DB), databaseFile);
                    this.loadDatabase(databaseFile);
                    this.startupMessages.add(TmmResourceBundle.getString("movie.loaddb.failed.restore"));
                    return;
                }
                catch (Exception e3) {
                    if (this.mvStore != null && !this.mvStore.isClosed()) {
                        this.mvStore.close();
                    }
                    LOGGER.error("Could not open database file from backup: {}", (Object)e3.getMessage());
                }
            }
            LOGGER.info("starting over with an empty database file");
            try {
                Utils.deleteFileSafely(databaseFile);
                this.loadDatabase(databaseFile);
                this.startupMessages.add(TmmResourceBundle.getString("movie.loaddb.failed"));
            }
            catch (Exception e1) {
                LOGGER.error("could not move old database file and create a new one: {}", (Object)e1.getMessage());
            }
            return;
        }
    }

    private void loadDatabase(final Path databaseFile) {
        Thread.UncaughtExceptionHandler exceptionHandler = new Thread.UncaughtExceptionHandler(){
            private int counter = 0;

            @Override
            public void uncaughtException(Thread t, Throwable e) {
                if (e instanceof IllegalStateException) {
                    if (this.counter < 10) {
                        ++this.counter;
                        return;
                    }
                    LOGGER.error("database corruption detected - try to recover");
                    MovieModuleManager.this.mvStore.close();
                    try {
                        Utils.deleteFileSafely(Paths.get(Globals.BACKUP_FOLDER, "movies.db.corrupted"));
                        Utils.moveFileSafe(databaseFile, Paths.get(Globals.BACKUP_FOLDER, "movies.db.corrupted"));
                    }
                    catch (Exception e1) {
                        LOGGER.error("Could not move corrupted database to '{}' - '{}", (Object)"movies.db.corrupted", (Object)e1.getMessage());
                    }
                    MovieModuleManager.this.mvStore = new MVStore.Builder().fileName(databaseFile.toString()).compressHigh().autoCommitBufferSize(MovieModuleManager.this.autoCommitBufferSize).backgroundExceptionHandler((Thread.UncaughtExceptionHandler)this).open();
                    MovieModuleManager.this.mvStore.setAutoCommitDelay(2000);
                    MovieModuleManager.this.mvStore.setRetentionTime(0);
                    MovieModuleManager.this.mvStore.setReuseSpace(true);
                    MovieModuleManager.this.mvStore.setCacheSize(8);
                    MovieModuleManager.this.movieMap = MovieModuleManager.this.mvStore.openMap("movies");
                    MovieModuleManager.this.movieSetMap = MovieModuleManager.this.mvStore.openMap("movieSets");
                    for (Movie movie : MovieModuleManager.this.getMovieList().getMovies()) {
                        MovieModuleManager.this.persistMovie(movie);
                    }
                    for (MovieSet movieSet : MovieModuleManager.this.getMovieList().getMovieSetList()) {
                        MovieModuleManager.this.persistMovieSet(movieSet);
                    }
                    this.counter = 0;
                }
            }
        };
        this.mvStore = new MVStore.Builder().fileName(databaseFile.toString()).compressHigh().autoCommitBufferSize(this.autoCommitBufferSize).backgroundExceptionHandler(exceptionHandler).open();
        this.mvStore.setAutoCommitDelay(2000);
        this.mvStore.setRetentionTime(0);
        this.mvStore.setReuseSpace(true);
        this.mvStore.setCacheSize(8);
        this.movieMap = this.mvStore.openMap("movies");
        this.movieSetMap = this.mvStore.openMap("movieSets");
        this.getMovieList().loadMoviesFromDatabase(this.movieMap);
        this.getMovieList().loadMovieSetsFromDatabase(this.movieSetMap);
        this.getMovieList().initDataAfterLoading();
    }

    @Override
    public synchronized void shutDown() throws Exception {
        if (!this.isEnabled()) {
            return;
        }
        this.enabled = false;
        this.databaseTimer.cancel();
        if (this.mvStore != null && !this.mvStore.isClosed()) {
            this.writePendingChanges(true);
            this.mvStore.commit();
            this.mvStore.compactMoveChunks();
            this.mvStore.close();
        }
        if (Settings.getInstance().isDeleteTrashOnExit()) {
            for (String ds : this.getSettings().getMovieDataSource()) {
                Path file = Paths.get(ds, ".deletedByTMM");
                Utils.deleteDirectoryRecursive(file);
            }
        }
    }

    private void writePendingChanges() {
        this.writePendingChanges(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void writePendingChanges(boolean force) {
        if (this.mvStore == null || this.mvStore.isClosed()) {
            return;
        }
        if (force) {
            this.lock.writeLock().lock();
        } else if (!this.lock.writeLock().tryLock()) {
            return;
        }
        try {
            HashMap<MediaEntity, Long> pending = new HashMap<MediaEntity, Long>(this.pendingChanges);
            long now = System.currentTimeMillis();
            for (Map.Entry entry : pending.entrySet()) {
                if (!force && (Long)entry.getValue() >= now - 2000L) continue;
                try {
                    MovieSet movieSet;
                    String newValue;
                    String oldValue;
                    if (entry.getKey() instanceof Movie) {
                        Movie movie = (Movie)entry.getKey();
                        oldValue = (String)this.movieMap.get((Object)movie.getDbId());
                        if (StringUtils.equals((CharSequence)oldValue, (CharSequence)(newValue = this.movieObjectWriter.writeValueAsString((Object)movie)))) continue;
                        this.movieMap.put((Object)movie.getDbId(), (Object)newValue);
                        continue;
                    }
                    if (!(entry.getKey() instanceof MovieSet) || StringUtils.equals((CharSequence)(oldValue = (String)this.movieSetMap.get((Object)(movieSet = (MovieSet)entry.getKey()).getDbId())), (CharSequence)(newValue = this.movieSetObjectWriter.writeValueAsString((Object)movieSet)))) continue;
                    this.movieSetMap.put((Object)movieSet.getDbId(), (Object)newValue);
                }
                catch (Exception e) {
                    LOGGER.warn("could not store '{}' - '{}'", (Object)((MediaEntity)entry.getKey()).getClass().getName(), (Object)e.getMessage());
                }
                finally {
                    this.pendingChanges.remove(entry.getKey());
                }
            }
        }
        finally {
            this.mvStore.commit();
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public boolean isEnabled() {
        return this.enabled;
    }

    public void dump(Movie movie) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            Object json = mapper.readValue((String)this.movieMap.get((Object)movie.getDbId()), Object.class);
            String s = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(json);
            LOGGER.info("Dumping Movie: {}\n{}", (Object)movie.getDbId(), (Object)s);
        }
        catch (Exception e) {
            LOGGER.error("Cannot parse JSON!", (Throwable)e);
        }
    }

    public void dump(MovieSet movieSet) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            Object json = mapper.readValue((String)this.movieSetMap.get((Object)movieSet.getDbId()), Object.class);
            String s = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(json);
            LOGGER.info("Dumping MovieSet: {}\n{}", (Object)movieSet.getDbId(), (Object)s);
        }
        catch (Exception e) {
            LOGGER.error("Cannot parse JSON!", (Throwable)e);
        }
    }

    void persistMovie(Movie movie) {
        try {
            this.lock.writeLock().lock();
            this.pendingChanges.put(movie, System.currentTimeMillis());
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    void removeMovieFromDb(Movie movie) {
        try {
            this.lock.writeLock().lock();
            this.pendingChanges.remove(movie);
            this.movieMap.remove((Object)movie.getDbId());
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    void persistMovieSet(MovieSet movieSet) {
        try {
            this.lock.writeLock().lock();
            this.pendingChanges.put(movieSet, System.currentTimeMillis());
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    void removeMovieSetFromDb(MovieSet movieSet) {
        try {
            this.lock.writeLock().lock();
            this.movieSetMap.remove((Object)movieSet.getDbId());
            this.pendingChanges.remove(movieSet);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void initializeDatabase() {
        Utils.deleteFileSafely(Paths.get(Settings.getInstance().getSettingsFolder(), MOVIE_DB));
    }

    @Override
    public void saveSettings() {
        this.getSettings().saveSettings();
    }

    @Override
    public List<String> getStartupMessages() {
        return this.startupMessages;
    }

    ObjectReader getMovieObjectReader() {
        return this.movieObjectReader;
    }

    ObjectReader getMovieSetObjectReader() {
        return this.movieSetObjectReader;
    }
}

